# -*- python -*-

### Copyright (C) 2005 Peter Williams <pwil3058@bigpond.net.au>

### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; version 2 of the License only.

### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.

### You should have received a copy of the GNU General Public License
### along with this program; if not, write to the Free Software
### Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# This file provides access to "quilt" functionality required by "gquilt"

import sys, os, os.path, re, pango

from gquilt_pkg import gquilt_tool
from gquilt_pkg import console, fsdb, ws_event, cmd_result
from gquilt_pkg import const
from gquilt_pkg import putils
from gquilt_pkg import utils

FSTATUS_MODIFIED = ' '
FSTATUS_ADDED = '+'
FSTATUS_REMOVED = '-'
FSTATUS_IGNORED = 'I'

PatchStatusMap = putils.StatusMap(extant=FSTATUS_MODIFIED, added=FSTATUS_ADDED, deleted=FSTATUS_REMOVED)

FSTATUS_MODIFIED_SET = set([FSTATUS_MODIFIED, FSTATUS_ADDED, FSTATUS_REMOVED])

class PatchDir(fsdb.GenDir):
    def __init__(self):
        fsdb.GenDir.__init__(self)
    def _new_dir(self):
        return PatchDir()
    def _update_own_status(self):
        if self.status_set & FSTATUS_MODIFIED_SET:
            self.status = FSTATUS_MODIFIED
    def _is_hidden_dir(self, dkey):
        status = self.subdirs[dkey].status
        if status not in [FSTATUS_MODIFIED]:
            return dkey[0] == '.' or status == FSTATUS_IGNORED
        return False
    def _is_hidden_file(self, fdata):
        if fdata.status not in FSTATUS_MODIFIED_SET:
            return fdata.name[0] == '.' or fdata.status == FSTATUS_IGNORED
        return False

class PatchFileDb(fsdb.GenFileDb):
    def __init__(self, file_list):
        fsdb.GenFileDb.__init__(self, PatchDir)
        for item in file_list:
            self.base_dir.add_file(item[2:].split(os.sep), item[0], None)

class QuiltCommands:
    def __init__(self):
        self.cmds = self._get_commands()
    def _get_commands(self):
        _res, sout, _serr = utils.run_cmd("quilt")
        if sout == "":
            return None
        lines = sout.splitlines()
        index = 2
        cmds = []
        while True:
            lcmds = lines[index].split()
            if len(lcmds) > 0:
                cmds += lcmds
            else:
                cmds.sort()
                return cmds
            index += 1
    def has_cmd(self, cmd):
        return cmd in self.cmds
    def get_list(self):
        return self.cmds

# Find the value of a specified quilt environment variable.  These can be set
# in as environment variables or in /etc/quilt.quiltrc, ~/.quiltrc or the file
# specified by the QUILTRC environment variable (if they exist).  Only one of
# the files will be read with precedence in the reverse of the order listed.
# Any variable set in the file that is read will override the value for the
# same variable that has been set as an environment variable.
# So we need to do the same so we know what quilt will see.
_DEFAULT_ENV_VAR_VALUE = { 'QUILT_PATCHES': 'patches', 'QUILT_SERIES': 'series' }

def _get_quilt_env_var_value(vname, default_ok=False):
    if default_ok:
        try:
            val = _DEFAULT_ENV_VAR_VALUE[vname]
        except LookupError:
            val = None
    else:
        val = None
    try:
        quiltrc = os.environ['QUILTRC']
    except LookupError:
        for fname in [os.path.expanduser("~/.quiltrc"), "/etc/quilt.quiltrc"]:
            if os.path.exists(fname):
                quiltrc = fname
                break
            else:
                quiltrc = None
    if quiltrc:
        regex = re.compile("^\w*%s=(.*)$" % vname)
        quiltrc_f = open(quiltrc, 'r')
        for line in quiltrc_f.readlines():
            m = regex.match(line.strip())
            if m:
                val = m.group(1)
                break
        quiltrc_f.close()
    else:
        try:
            val = os.environ[vname]
        except LookupError:
            pass
    return val

def find_patchfns():
    binloc = utils.which("quilt")
    if binloc is None:
        return None
    bindex = binloc.find("bin" + os.path.sep + "quilt")
    if bindex == -1:
        return None
    res = os.path.join(binloc[0:bindex], "share", "quilt", "scripts", "patchfns")
    if not os.path.exists(res) or not os.path.isfile(res):
        return None
    return res

def _find_gquilt_lib_dir():
    try:
        result = os.environ['GQUILT_LIB_DIR']
    except LookupError:
        result = None
        for path in sys.path:
            tempth = os.path.join(path, 'gquilt_pkg')
            if os.path.exists(tempth) and os.path.isdir(tempth):
                result = tempth
                break
    return result

_QBSFE_FILE_NAME = os.path.join(os.sep, 'etc', 'gquilt.d', 'qbsfe' + os.extsep + 'sh')
if not os.path.exists(_QBSFE_FILE_NAME):
    _QBSFE_FILE_NAME = os.path.join(_find_gquilt_lib_dir(), 'qbsfe' + os.extsep + 'sh')
    if not os.path.exists(_QBSFE_FILE_NAME):
        raise Exception('Could not find "qbsfe.sh"')

_PATCHFNS = find_patchfns()
if _PATCHFNS is None:
    _QBSFE = None
else:
    _QBSFE = 'bash -c ". ' + _QBSFE_FILE_NAME + '" gquilt ' + _PATCHFNS

class QuiltError(Exception):
    """A quilt specific error"""

def internal(cmd, cmd_input=None):
    if _QBSFE is None:
        raise QuiltError("Couldn't find patchfuns")
    return utils.run_cmd(" ".join([_QBSFE, cmd]), cmd_input)

# Some useful private quilt functions (that shouldn't fail)
def _find_patch(patch):
    res, sout, serr = internal(" ".join(["find_patch", patch]))
    if res != 0:
        raise QuiltError(serr)
    return sout[:-1]

def _patch_file_name(patch):
    res, sout, serr = internal(" ".join(["patch_file_name", patch]))
    if res != 0:
        raise QuiltError(serr)
    return sout[:-1]

def _patch_basename(patchfile):
    res, sout, serr = internal(" ".join(["basename", patchfile]))
    if res != 0:
        raise QuiltError(serr)
    return sout[:-1]

def _convert_pop_res(res, sout, serr):
    if res != cmd_result.OK:
        if re.search("needs to be refreshed first", sout + serr):
            res |= cmd_result.SUGGEST_REFRESH
        elif re.search("\(refresh it or enforce with -f\)", sout + serr):
            res |= cmd_result.SUGGEST_FORCE_OR_REFRESH
    return cmd_result.Result(res, sout, serr)

def _convert_push_res(res, sout, serr):
    if res != cmd_result.OK:
        if re.search("\(forced; needs refresh\)", sout + serr):
            res = cmd_result.OK
        elif re.search("needs to be refreshed first", sout + serr):
            res |= cmd_result.SUGGEST_REFRESH
        elif re.search("\(enforce with -f\)", sout + serr):
            res |= cmd_result.SUGGEST_FORCE
    return cmd_result.Result(res, sout, serr)

def _convert_import_res(res, sout, serr):
    if res != cmd_result.OK and re.search("Replace with -f", sout + serr):
        res |= cmd_result.SUGGEST_FORCE_OR_RENAME
    return cmd_result.Result(res, sout, serr)

def _quilt_version():
    res, sout, _serr = utils.run_cmd("quilt --version")
    if res != 0:
        return None
    return sout

def _quilt_version_ge(rmajor, rminor):
    vstr = _quilt_version()
    if not vstr:
        return False
    vstrs = vstr.split(".")
    amajor = int(vstrs[0])
    if  amajor > rmajor:
        return True
    elif amajor < rmajor:
        return False
    else:
        return int(vstrs[1]) >= rminor

def _patch_dir(fdir=None):
    pdir = _get_quilt_env_var_value('QUILT_PATCHES', default_ok=True)
    if fdir is None:
        fdir = os.getcwd()
    return os.path.join(fdir, pdir)
    
def _series_file(fdir=None):
    sname = _get_quilt_env_var_value('QUILT_SERIES', default_ok=True)
    return os.path.join(_patch_dir(fdir), sname)
    
# Now implement the tool interface for quilt
class Interface(gquilt_tool.Interface):
    def __init__(self):
        gquilt_tool.Interface.__init__(self, "quilt")
        self.status_deco_map = {
            None: gquilt_tool.Deco(pango.STYLE_NORMAL, "black"),
            FSTATUS_MODIFIED: gquilt_tool.Deco(pango.STYLE_NORMAL, "blue"),
            FSTATUS_ADDED: gquilt_tool.Deco(pango.STYLE_NORMAL, "darkgreen"),
            FSTATUS_REMOVED: gquilt_tool.Deco(pango.STYLE_NORMAL, "red"),
            FSTATUS_IGNORED: gquilt_tool.Deco(pango.STYLE_ITALIC, "grey"),
        }
    def _map_cmd_result(self, result, ignore_err_re=None):
        if result.eflags == 0:
            return cmd_result.map_cmd_result(result, ignore_err_re=ignore_err_re)
        else:
            flags = cmd_result.ERROR
            return cmd_result.Result(flags, result.stdout, result.serr)
    def display_files_diff_in_viewer(self, viewer, files, patch=None):
        difffld = "--diff=" + viewer
        if patch is None:
            pid = os.spawnlp(os.P_NOWAIT, "quilt", "quilt", "diff", difffld, files[0])
        else:
            pid = os.spawnlp(os.P_NOWAIT, "quilt", "quilt", "diff", difffld, "-P", patch, files[0])
    def do_add_files_to_patch(self, filelist):
        cmd = "quilt add"
        # quilt will screw up semi silently if you try to add directories to a patch
        for f in filelist:
            if os.path.isdir(f):
                ws_event.notify_events(ws_event.FILE_ADD)
                return cmd_result.Result(cmd_result.ERROR, "", f + ' is a directory.\n "quilt" does not handle adding directories to patches\n')
        result = console.exec_console_cmd(" ".join([cmd, " ".join(filelist)]))
        ws_event.notify_events(ws_event.FILE_ADD)
        return result
    def do_delete_patch(self, patch):
        result = console.exec_console_cmd(" ".join(["quilt", "delete", patch]))
        ws_event.notify_events(ws_event.PATCH_DELETE)
        return result
    def do_exec_tool_cmd(self, cmd):
        result = console.exec_console_cmd(" ".join(["quilt", cmd]))
        ws_event.notify_events(ws_event.FILE_MOD|ws_event.FILE_ADD|ws_event.ALL_EVENTS)
        return result
    def do_finish_patch(self, patch):
        return cmd_result.Result(cmd_result.ERROR, "", '"quilt" does not have this feature')
    def do_fold_patch(self, patch):
        filename = self.get_patch_file_name(patch)
        res, sout, serr = self.do_fold_patch_file(filename)
        if res == 0:
            self.do_delete_patch(patch)
        return cmd_result.Result(res, sout, serr)
    def do_fold_patch_file(self, filename):
        result = console.exec_console_cmd(" < ".join(["quilt fold", filename]))
        ws_event.notify_events(ws_event.FILE_CHANGES)
        return result
    def do_import_patch(self, filename, patchname=None, force=False):
        if patchname:
            cmd = "quilt import -P %s " % patchname
        else:
            cmd = "quilt import "
        if force:
            cmd += "-f -d n "
        res, sout, serr = console.exec_console_cmd(cmd + filename)
        ws_event.notify_events(ws_event.PATCH_CREATE)
        return _convert_import_res(res, sout, serr)
    def do_new_patch(self, name, force=False):
        if force:
            return cmd_result.Result(cmd_result.ERROR, "", "\"quilt\" cannot force \"new\"")
        result = console.exec_console_cmd(" ".join(["quilt", "new", name]))
        ws_event.notify_events(ws_event.PATCH_CREATE|ws_event.PATCH_PUSH)
        return result
    def do_pop_to(self, patch=None):
        cmd = "quilt pop"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        res, sout, serr = console.exec_console_cmd(cmd)
        events = ws_event.PATCH_POP
        if not self.get_in_progress():
            events |= ws_event.PMIC_CHANGE
        ws_event.notify_events(events)
        return _convert_pop_res(res, sout, serr)
    def do_push_to(self, patch=None, force=False, merge=False):
        in_charge = self.get_in_progress()
        cmd = "quilt push"
        if force:
            cmd += " -f"
        if patch is not None:
            if patch is "":
                cmd += " -a"
            else:
                cmd += " " + patch
        res, sout, serr = console.exec_console_cmd(cmd)
        events = ws_event.PATCH_PUSH
        if not in_charge:
            events |= ws_event.PMIC_CHANGE
        ws_event.notify_events(events)
        return _convert_push_res(res, sout, serr)
    def do_refresh(self, patch=None, force=False, notify=True):
        cmd = "quilt refresh"
        if force:
            cmd += " -f"
        if patch is not None:
            cmd = " ".join([cmd, patch])
        res, sout, serr = console.exec_console_cmd(cmd)
        if res != cmd_result.OK:
            if re.search("Enforce refresh with -f", sout + serr):
                res = cmd_result.ERROR_SUGGEST_FORCE
            else:
                res = cmd_result.ERROR
        if notify:
            ws_event.notify_events(ws_event.PATCH_REFRESH)
        return cmd_result.Result(res, sout, serr)
    def do_remove_files_from_patch(self, filelist, patch=None):
        if QuiltCommands().has_cmd("remove"):
            if patch == None:
                cmd = "quilt remove"
            else:
                if _quilt_version_ge(0, 43):
                    cmd = "quilt remove -P " + patch
                else:
                    cmd = "quilt remove -p " + patch
            # quilt will screw up semi silently if you try to remove directories from a patch
            for f in filelist:
                if os.path.isdir(f):
                    return cmd_result.Result(cmd_result.ERROR, "", f + ' is a directory.\n "quilt" does not handle removing directories from patches\n')
            result = console.exec_console_cmd(" ".join([cmd, " ".join(filelist)]))
        else:
            result = (cmd_result.ERROR, "", '"quilt" does not handle removing files from patches\n')
        ws_event.notify_events(ws_event.FILE_MOD|ws_event.FILE_ADD|ws_event.FILE_DEL)
        return result
    def do_rename_patch(self, patch, newname):
        cmd = "quilt rename "
        if patch is not None:
            if _quilt_version_ge(0, 43):
                cmd += "-P " + patch + " "
            else:
                cmd += "-p " + patch + " "
        cmd += newname
        res, sout, serr = console.exec_console_cmd(cmd)
        ws_event.notify_events(ws_event.PATCH_CREATE|ws_event.PATCH_DELETE)
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        else:
            return cmd_result.Result(cmd_result.OK, sout, serr)
    def do_revert_files_in_patch(self, filelist, patch=None):
        if QuiltCommands().has_cmd("revert"):
            if patch == None:
                cmd = "quilt revert"
            else:
                cmd = "quilt revert -P " + patch
            # quilt will screw up semi silently if you try to revert directories in a patch
            for f in filelist:
                if os.path.isdir(f):
                    return cmd_result.Result(cmd_result.ERROR, "", f + ' is a directory.'
                            + os.linesep + '"quilt" does not handle reversion for directories in patches' + os.linesep)
            result = console.exec_console_cmd(" ".join([cmd, " ".join(filelist)]))
        else:
            result = (cmd_result.ERROR, "", '"quilt" does not handle reverting changes to in from patches\n')
        ws_event.notify_events(ws_event.FILE_MOD|ws_event.FILE_ADD|ws_event.FILE_DEL)
        return result
    def do_select_guards(self, guards):
        return cmd_result.Result(cmd_result.ERROR, '', 'quilt does not support guards')
    def do_set_patch_guards(self, patch_name, guards):
        return cmd_result.Result(cmd_result.ERROR, '', 'quilt does not support guards')
    def extdiff_and_full_patch_ok(self):
        return False
    def get_all_patches_data(self):
        output = []
        res, sout, serr = utils.run_cmd('quilt series -v')
        for line in sout.splitlines():
            pname = line[2:]
            if line[0] == " ":
                output.append(gquilt_tool.PatchData(pname, const.NOT_APPLIED, []))
            else:
                output.append(gquilt_tool.PatchData(pname, const.APPLIED, []))
        return output
    def get_applied_patches(self):
        res, sout, err = utils.run_cmd('quilt applied')
        if res != 0:
            return []
        return sout.splitlines()
    def get_combined_diff(self, start_patch=None, end_patch=None):
        cmd = "quilt diff --sort --combine "
        if start_patch is None:
            cmd += "-"
        else:
            cmd += start_patch
        if end_patch is not None:
            cmd += " -P " + end_patch
        res, sout, serr = utils.run_cmd(cmd)
        if res != 0:
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        return cmd_result.Result(res, sout, serr)
    def get_diff(self, filelist=list(), patch=None):
        if patch is None:
            cmd = "quilt diff"
        else:
            cmd = "quilt diff -P " + patch
        res, sout, serr = utils.run_cmd(" ".join([cmd, " ".join(filelist)]))
        if res != 0:
            if not self.is_patch_applied(patch):
                res, diff = putils.get_patch_diff_lines(self.get_patch_file_name(patch))
                if res:
                    return cmd_result.Result(cmd_result.OK, os.linesep.join(diff), [])
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        return cmd_result.Result(res, sout, serr)
    def get_diff_for_files(self, file_list=None, patch=None):
        return self.get_diff(filelist=file_list, patch=patch)
    def get_in_progress(self):
        return self.get_top_patch() is not ''
    def get_next_patch(self):
        res, sout, serr = utils.run_cmd("quilt next")
        if res == 0 or (serr.strip() == "" and sout.strip() == ""):
            return sout.strip()
        elif (res == 512 or res == 2) and sout.strip() == "":
            return ""
        else:
            raise QuiltError(serr)
    def get_patch_file_db(self, patch=None):
        if patch and not self.is_patch_applied(patch):
            pfn = self.get_patch_file_name(patch)
            return putils.get_patch_file_db(pfn, PatchStatusMap)
        top = self.get_top_patch()
        if not top:
            # either we're not in an mq playground or no patches are applied
            return PatchFileDb([])
        cmd = 'quilt files -v'
        if patch is not None:
            cmd += " " + patch
        res, sout, serr = utils.run_cmd(cmd)
        return PatchFileDb(sout.splitlines())
    def get_patch_file_name(self, patch):
        return _patch_file_name(_find_patch(patch))
    def get_patch_files(self, patch=None, withstatus=True):
        cmd = "quilt files"
        if withstatus:
            cmd += " -v"
        if patch is not None:
            cmd += " " + patch
        elif self.get_top_patch() == "":
            return cmd_result.Result(cmd_result.OK, "", "")
        res, sout, serr = utils.run_cmd(cmd)
        if res != 0:
            if patch and not self.is_patch_applied(patch):
                patchfile = self.get_patch_file_name(patch)
                is_ok, filelist = putils.get_patch_files(patchfile, withstatus)
                if is_ok:
                    return cmd_result.Result(cmd_result.OK, filelist, "")
                else:
                    return cmd_result.Result(cmd_result.ERROR, "", filelist)
            return cmd_result.Result(cmd_result.ERROR, sout, serr)
        if withstatus:
            filelist = []
            for line in sout.splitlines():
                if line[0] == "+":
                    filelist.append((line[2:], const.ADDED))
                elif line[0] == "-":
                    filelist.append((line[2:], const.DELETED))
                else:
                    filelist.append((line[2:], const.EXTANT))
        else:
            filelist = sout.splitlines()
        return cmd_result.Result(res, filelist, serr)
    def get_patch_guards(self, patch):
        return []
    def get_playground_root(self, fdir=None):
        pdir = _get_quilt_env_var_value('QUILT_PATCHES', default_ok=True)
        if not fdir:
            fdir = os.getcwd()
        root = fdir
        while True:
            apd = os.path.join(root, pdir)
            if os.path.exists(apd) and os.path.isdir(apd):
                return root
            newroot = os.path.dirname(root)
            if root == newroot:
                break
            root = newroot
        return None
    def get_selected_guards(self):
        return []
    def get_top_patch(self):
        res, sout, serr = utils.run_cmd("quilt top")
        if res == 0 or (serr.strip() == "" and sout.strip() == ""):
            return sout.strip()
        elif (res == 512 or res == 2) and sout.strip() == "":
            return ""
        else:
            raise QuiltError(serr)
    def get_ws_file_db(self):
        return fsdb.OsFileDb()
    def has_add_files(self):
        return True
    def has_finish_patch(self):
        return False
    def has_refresh_non_top(self):
        return True
    def is_available(self):
        if _quilt_version() is None:
            return False
        return True
    def is_patch_applied(self, patch):
        res, sout, serr = internal("is_applied " + patch)
        return res == 0
    def is_playground(self, fdir=None):
        root = self.get_playground_root(fdir)
        if not root:
            return False
        return os.path.exists(_series_file(root))
    def last_patch_in_series(self):
        res, sout, serr = utils.run_cmd("quilt series")
        if res != 0:
            raise QuiltError(serr)
        return sout.splitlines()[-1]
    def new_playground(self, fdir=None):
        patch_dir = _patch_dir(fdir)
        if not os.path.exists(patch_dir):
            try:
                os.makedirs(patch_dir)
            except os.error as value:
                return cmd_result.Result(cmd_result.ERROR, "", value[1])
        return utils.run_cmd("touch " + _series_file(fdir))
    def requires(self):
        return "\"quilt\" <http://savannah.nongnu.org/projects/quilt>"
