/*
 * @(#)ClassLoadHelperUTest.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */

package net.sourceforge.groboutils.util.classes.v1;

import org.easymock.EasyMock;
import org.easymock.MockControl;
import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;


/**
 * Tests the ClassLoadHelper class.
 *
 * @author    Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version   $Date: 2003/02/10 22:52:38 $
 * @since     March 1, 2002
 */
public class ClassLoadHelperUTest extends TestCase
{
    //-------------------------------------------------------------------------
    // Standard JUnit Class-specific declarations
    
    private static final Class THIS_CLASS = ClassLoadHelperUTest.class;
    
    public ClassLoadHelperUTest( String name )
    {
        super( name );
    }

    
    //-------------------------------------------------------------------------
    // setup
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void setUp() throws Exception
    {
        super.setUp();
        
        // set ourself up
    }


    //-------------------------------------------------------------------------
    // Tests
    
    
    public void testConstructor1()
    {
        new ClassLoadHelper();
    }
    
    
    public void testConstructor2()
    {
        try
        {
            new ClassLoadHelper( (Class)null );
            fail("Did not throw NullPointerException");
        }
        catch (NullPointerException npe)
        {
            // check exception?
        }
    }
    
    
    public void testConstructor2a()
    {
        new ClassLoadHelper( (ClassLoader)null );
    }
    
    
    public void testConstructor3()
    {
        new ClassLoadHelper( createClassLoader() );
    }
    
    
    public void testGetClass1()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        Class c = clh.getClass( null );
        assertNull(
            "Null string should return null.",
            c );
    }
    
    
    private static final String BAD_CLASS_NAME_1 = "";
    // totally wrong class name - can never exist due to naming rules.
    private static final String BAD_CLASS_NAME_2 = "java.lang.2-Not A Class";
    
    public void testGetClass2()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        Class c = clh.getClass( BAD_CLASS_NAME_1 );
        assertNull(
            "Bad class name should return null.",
            c );
    }
    
    public void testGetClass3()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        Class c = clh.getClass( BAD_CLASS_NAME_2 );
        assertNull(
            "Bad class name should return null.",
            c );
    }
    
    
    public void testGetClass4()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        Class c = clh.getClass( String.class.getName() );
        assertEquals(
            "Should have returned the String class with the same classloader "+
            "as the one that loaded ourself.",
            String.class,
            c );
    }
    
    
    public interface MyInterface {}
    public class MyInnerClass {}
    public static class MyGoodClass {}
    
    public void testCreateObjectS1()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        Object o = clh.createObject( (String)null );
        assertNull(
            "Null string should return null.",
            o );
    }
    
    public void testCreateObjectS2()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that doesn't exist
        Object o = clh.createObject( BAD_CLASS_NAME_1 );
        assertNull(
            "Bad class name should return null.",
            o );
    }
    
    public void testCreateObjectS3()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that doesn't exist
        Object o = clh.createObject( BAD_CLASS_NAME_2 );
        assertNull(
            "Bad class name should return null.",
            o );
    }
    
    public void testCreateObjectS4()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that can't be instantiated
        Object o = clh.createObject( MyInterface.class.getName() );
        assertNull(
            "Interface should return null.",
            o );
    }
    
    public void testCreateObjectS5()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that can't be instantiated from this context
        Object o = clh.createObject( MyInnerClass.class.getName() );
        assertNull(
            "Inner class should return null.",
            o );
    }
    
    public void testCreateObjectS6()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that can't be instantiated due to no default constructor
        Object o = clh.createObject( this.getClass().getName() );
        assertNull(
            "No-default constructor class should return null.",
            o );
    }
    
    public void testCreateObjectS7()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        Object o = clh.createObject( MyGoodClass.class.getName() );
        assertNotNull(
            "Good class should not return null.",
            o );
        assertEquals(
            "Good class should be of the correct class.",
            MyGoodClass.class,
            o.getClass() );
    }

    
    public void testCreateObjectSZ1()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // does not throw exception
        Object o = clh.createObject( (String)null, false );
        assertNull(
            "Null string should return null.",
            o );
    }
    
    public void testCreateObjectSZ2()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that doesn't exist
        try
        {
            Object o = clh.createObject( BAD_CLASS_NAME_1, false );
            fail( "Did not throw an IllegalStateException: retrieved "+o );
        }
        catch (IllegalStateException ise)
        {
            // inspect exception?
        }
    }
    
    public void testCreateObjectSZ3()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that doesn't exist
        try
        {
            Object o = clh.createObject( BAD_CLASS_NAME_2, false );
            fail( "Did not throw an IllegalStateException: retrieved "+o );
        }
        catch (IllegalStateException ise)
        {
            // inspect exception?
        }
    }
    
    public void testCreateObjectSZ4()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that can't be instantiated
        try
        {
            Object o = clh.createObject( MyInterface.class.getName(), false );
            fail( "Did not throw an IllegalStateException: retrieved "+o );
        }
        catch (IllegalStateException ise)
        {
            // inspect exception?
        }
    }
    
    public void testCreateObjectSZ5()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that can't be instantiated from this context
        try
        {
            Object o = clh.createObject( MyInnerClass.class.getName(), false );
            fail( "Did not throw an IllegalStateException: retrieved "+o );
        }
        catch (IllegalStateException ise)
        {
            // inspect exception?
        }
    }
    
    public void testCreateObjectSZ6()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        // Class that can't be instantiated due to no default constructor
        try
        {
            Object o = clh.createObject( this.getClass().getName(), false );
            fail( "Did not throw an IllegalStateException." );
        }
        catch (IllegalStateException ise)
        {
            // inspect exception?
        }
    }
    
    public void testCreateObjectSZ7()
    {
        ClassLoadHelper clh = new ClassLoadHelper();
        
        Object o = clh.createObject( MyGoodClass.class.getName(), false );
        assertNotNull(
            "Good class should not return null.",
            o );
        assertEquals(
            "Good class should be of the correct class.",
            MyGoodClass.class,
            o.getClass() );
    }
    
    
    
    
    //-------------------------------------------------------------------------
    // Helpers
    
    
    protected ClassLoader createClassLoader()
    {
        return new ClassLoader() {
            public Class loadClass( String s, boolean f )
            {
                return null;
            }
        };
    }
    
    
    //-------------------------------------------------------------------------
    // Standard JUnit declarations
    
    
    public static Test suite()
    {
        TestSuite suite = new TestSuite( THIS_CLASS );
        
        // Test the implementation's interface conformity.
        /*
        suite.addTest( IxUTestI.suite(
            new ImplementationCreator[] {
                new ImplementationCreator() {
                    public Object createImplementedObject() {
                        // XXXXXXXXXXXXXXXXXXXXXXXX
                    }
                },
            } ) );
        */
        
        return suite;
    }
    
    public static void main( String[] args )
    {
        String[] name = { THIS_CLASS.getName() };
        
        // junit.textui.TestRunner.main( name );
        // junit.swingui.TestRunner.main( name );
        
        junit.textui.TestRunner.main( name );
    }
    
    
    /**
     * 
     * @exception Exception thrown under any exceptional condition.
     */
    protected void tearDown() throws Exception
    {
        // tear ourself down
        
        
        super.tearDown();
    }
}

