package org.jboss.cache.interceptors;

import org.jboss.cache.Fqn;
import org.jboss.cache.InvocationContext;
import org.jboss.cache.Modification;
import org.jboss.cache.NodeSPI;
import org.jboss.cache.commands.read.GetChildrenNamesCommand;
import org.jboss.cache.commands.read.GetKeyValueCommand;
import org.jboss.cache.commands.read.GetKeysCommand;
import org.jboss.cache.commands.read.GetNodeCommand;
import org.jboss.cache.commands.tx.OptimisticPrepareCommand;
import org.jboss.cache.commands.tx.PrepareCommand;
import org.jboss.cache.commands.write.ClearDataCommand;
import org.jboss.cache.commands.write.MoveCommand;
import org.jboss.cache.commands.write.PutDataMapCommand;
import org.jboss.cache.commands.write.PutForExternalReadCommand;
import org.jboss.cache.commands.write.PutKeyValueCommand;
import org.jboss.cache.commands.write.RemoveKeyCommand;
import org.jboss.cache.commands.write.RemoveNodeCommand;
import org.jboss.cache.factories.annotations.Inject;
import org.jboss.cache.transaction.GlobalTransaction;
import org.jboss.cache.transaction.TransactionEntry;

import javax.transaction.SystemException;
import javax.transaction.TransactionManager;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;

/**
 * Loads nodes that don't exist at the time of the call into memory from the CacheLoader.
 * If the nodes were evicted earlier then we remove them from the cache loader after
 * their attributes have been initialized and their children have been loaded in memory.
 *
 * @author <a href="mailto:{hmesha@novell.com}">{Hany Mesha}</a>
 * @version $Id: ActivationInterceptor.java 7733 2009-02-19 13:21:22Z manik.surtani@jboss.com $
 */
public class ActivationInterceptor extends CacheLoaderInterceptor implements ActivationInterceptorMBean
{

   protected TransactionManager txMgr = null;
   private long activations = 0;

   /**
    * List<Transaction> that we have registered for
    */
   protected ConcurrentHashMap transactions = new ConcurrentHashMap(16);
   protected static final Object NULL = new Object();

   public ActivationInterceptor()
   {
      isActivation = true;
      useCacheStore = false;
   }

   @Inject
   public void injectTransactionManager(TransactionManager txMgr)
   {
      this.txMgr = txMgr;
   }

   @Override
   public Object visitClearDataCommand(InvocationContext ctx, ClearDataCommand command) throws Throwable
   {
      Object returnValue = super.visitClearDataCommand(ctx, command);
      if (trace)
         log.trace("This is a remove data operation; removing the data from the loader, no activation processing needed.");
      loader.removeData(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitRemoveNodeCommand(InvocationContext ctx, RemoveNodeCommand command) throws Throwable
   {
      Object returnValue = super.visitRemoveNodeCommand(ctx, command);
      if (trace)
         log.trace("This is a remove operation; removing the node from the loader, no activation processing needed.");
      loader.remove(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitGetChildrenNamesCommand(InvocationContext ctx, GetChildrenNamesCommand command) throws Throwable
   {
      Object returnValue = super.visitGetChildrenNamesCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitGetKeysCommand(InvocationContext ctx, GetKeysCommand command) throws Throwable
   {
      Object returnValue = super.visitGetKeysCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitGetNodeCommand(InvocationContext ctx, GetNodeCommand command) throws Throwable
   {
      Object returnValue = super.visitGetNodeCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitGetKeyValueCommand(InvocationContext ctx, GetKeyValueCommand command) throws Throwable
   {
      Object returnValue = super.visitGetKeyValueCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitPutForExternalReadCommand(InvocationContext ctx, PutForExternalReadCommand command) throws Throwable
   {
      return visitPutKeyValueCommand(ctx, command);
   }

   @Override
   public Object visitPutKeyValueCommand(InvocationContext ctx, PutKeyValueCommand command) throws Throwable
   {
      Object returnValue = super.visitPutKeyValueCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitPutDataMapCommand(InvocationContext ctx, PutDataMapCommand command) throws Throwable
   {
      Object returnValue = super.visitPutDataMapCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitRemoveKeyCommand(InvocationContext ctx, RemoveKeyCommand command) throws Throwable
   {
      Object returnValue = super.visitRemoveKeyCommand(ctx, command);
      removeNodeFromCacheLoader(command.getFqn());
      return returnValue;
   }

   @Override
   public Object visitMoveCommand(InvocationContext ctx, MoveCommand command) throws Throwable
   {
      Object returnValue = super.visitMoveCommand(ctx, command);
      if (trace)
         log.trace("This is a move operation; removing the FROM node from the loader, no activation processing needed.");
      loader.remove(command.getFqn());
      removeNodeFromCacheLoader(command.getFqn().getParent());
      removeNodeFromCacheLoader(command.getTo());
      return returnValue;
   }


   /**
    * Remove the node from the cache loader if it exists in memory,
    * its attributes have been initialized, its children have been loaded,
    * AND it was found in the cache loader (nodeLoaded = true).
    * Then notify the listeners that the node has been activated.
    */
   private void removeNodeFromCacheLoader(Fqn fqn) throws Throwable
   {
      NodeSPI n;
      if (((n = dataContainer.peek(fqn, true, false)) != null) && n.isDataLoaded() && loader.exists(fqn))
      {
         // node not null and attributes have been loaded?
         if (!n.getChildrenDirect().isEmpty())
         {
            boolean result = childrenLoaded(n);
            if (result)
            {
               log.debug("children all initialized");
               remove(fqn);
            }
         }
         else if (loaderNoChildren(fqn))
         {
            if (log.isDebugEnabled()) log.debug("no children " + n);
            remove(fqn);
         }
      }
   }

   private static boolean childrenLoaded(NodeSPI<?, ?> node)
   {
      if (!node.isChildrenLoaded())
      {
         return false;
      }
      for (NodeSPI child : node.getChildrenDirect())
      {
         if (!child.isDataLoaded())
         {
            return false;
         }
      }
      return true;

   }

   @Override
   public Object visitOptimisticPrepareCommand(InvocationContext ctx, OptimisticPrepareCommand command) throws Throwable
   {
      Object retval = invokeNextInterceptor(ctx, command);
      if (inTransaction())
      {
         prepareCacheLoader(ctx);
      }
      return retval;
   }

   private boolean inTransaction() throws SystemException
   {
      return txMgr != null && txMgr.getTransaction() != null;
   }

   @Override
   public Object visitPrepareCommand(InvocationContext ctx, PrepareCommand command) throws Throwable
   {
      Object retval = invokeNextInterceptor(ctx, command);
      if (inTransaction())
      {
         prepareCacheLoader(ctx);
      }
      return retval;
   }

   private void remove(Fqn fqn) throws Exception
   {
      loader.remove(fqn);
      if (getStatisticsEnabled()) activations++;
   }

   /**
    * Returns true if the loader indicates no children for this node.
    * Return false on error.
    */
   private boolean loaderNoChildren(Fqn fqn)
   {
      try
      {
         Set childrenNames = loader.getChildrenNames(fqn);
         return (childrenNames == null);
      }
      catch (Exception e)
      {
         log.error("failed getting the children names for " + fqn + " from the cache loader", e);
         return false;
      }
   }

   public long getActivations()
   {
      return activations;
   }

   @Override
   public void resetStatistics()
   {
      super.resetStatistics();
      activations = 0;
   }

   @Override
   public Map<String, Object> dumpStatistics()
   {
      Map<String, Object> retval = super.dumpStatistics();
      if (retval == null)
      {
         retval = new HashMap<String, Object>();
      }
      retval.put("Activations", activations);
      return retval;
   }

   private void prepareCacheLoader(InvocationContext ctx) throws Throwable
   {
      GlobalTransaction gtx = ctx.getGlobalTransaction();
      TransactionEntry entry = ctx.getTransactionEntry();
      if (entry == null)
      {
         throw new Exception("entry for transaction " + gtx + " not found in transaction table");
      }
      List<Modification> cacheLoaderModifications = new ArrayList<Modification>();

      if (cacheLoaderModifications.size() > 0)
      {
         loader.prepare(gtx, cacheLoaderModifications, false);
      }
   }
}
